/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.external.api.rest;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.List;

import javax.servlet.Servlet;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.eclipse.lsat.external.api.LSATException;
import org.eclipse.lsat.external.api.LSATService;
import org.eclipse.lsat.external.api.LSATService.FileType;
import org.eclipse.lsat.external.api.LSATService.TimingAnalysisGoal;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.http.whiteboard.HttpWhiteboardConstants;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

@Component(service = Servlet.class,
        property =
        {HttpWhiteboardConstants.HTTP_WHITEBOARD_SERVLET_PATTERN + "=/api/*",
                HttpWhiteboardConstants.HTTP_WHITEBOARD_CONTEXT_SELECT + "=("
                        + HttpWhiteboardConstants.HTTP_WHITEBOARD_CONTEXT_NAME + "=default)"})
@SuppressWarnings("unused")
public class LSATServiceServlet extends HttpServlet {
    private static final long serialVersionUID = 1L;

    private static final String APPLICATION_JSON = "application/json";

    private static final String TEXT_PLAIN = "text/plain";

    private static final String TEXT_HTML = "text/html";

    private static final String APPLICATION_XML = "application/xml";

    private Gson gson = new GsonBuilder().setPrettyPrinting().create();

    @Reference
    private LSATService lsatService;

    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
            throws ServletException, IOException
    {
        String pathInfo = request.getPathInfo();

        try {
            switch (pathInfo) {
                case "/timinganalysis" -> handleTimingAnalysis(request, response);
                case "/health" -> handleHealth(response);
                case "/list" -> handleList(request, response);
                case "/reload"-> handleReloadProject(request, response);
                case "/reloadall" -> handleReloadAllProjects(request, response);
                default -> response.sendError(HttpServletResponse.SC_NOT_FOUND);
            }
        } catch (LSATException e) {
            error(response, e);
        }
    }

    private void handleHealth(HttpServletResponse response) throws IOException {
        response.setContentType("text");
        PrintWriter writer = response.getWriter();
        writer.write("REST Health OK");
        writer.close();
    }

    private void handleList(HttpServletRequest request, HttpServletResponse response)
            throws IOException, LSATException
    {
        var pars = new Pararameters(request);
        String projectName = pars.stringValue("project");
        FileType fileType = FileType.valueOf(pars.stringValue("type").toUpperCase());

        List<String> result = lsatService.list(projectName, fileType);

        response.setContentType(APPLICATION_JSON);
        try (PrintWriter writer = response.getWriter()) {
            gson.toJson(result, writer);
        }
    }

    private void handleReloadProject(HttpServletRequest request, HttpServletResponse response)
            throws IOException, LSATException
    {
        var pars = new Pararameters(request);
        String projectName = pars.stringValue("project");
        String result = lsatService.reloadProject(projectName);

        response.setContentType("text");
        PrintWriter writer = response.getWriter();
        if (result != null) {
            writer.write("Successfully reloaded project: " + projectName);
        } else {
            writer.write("Failed to reload project: " + projectName);
        }
        writer.close();
    }

    private void handleReloadAllProjects(HttpServletRequest request, HttpServletResponse response)
            throws IOException, LSATException
    {
        List<String> result = lsatService.reloadAllProjects();

        response.setContentType("text");
        PrintWriter writer = response.getWriter();
        writer.write("Reloaded projects:\n");
        for (var r: result) {
            writer.write(" - " + r + "\n");
        }
        writer.close();
    }

    private void handleTimingAnalysis(HttpServletRequest request, HttpServletResponse response)
            throws IOException, LSATException
    {
        var pars = new Pararameters(request);
        String projectName = pars.stringValue("project");
        String dispatchFilename = pars.stringValue("dispatch");
        String settingFilename = pars.stringValue("setting");

        TimingAnalysisGoal goal = TimingAnalysisGoal
                .valueOf(pars.stringValue("goal", TimingAnalysisGoal.NORMAL.name()).toUpperCase());
        int stochasticSampleLength = pars.intValue("sampleLength", 100);
        boolean exportMotionPlots = pars.booleanValue("exportMotionPlots", false);
        int motionPlotSampleFrequency = pars.intValue("motionPlotSampleFrequency", 1000);
        String motionPlotFilter = pars.stringValue("motionPlotFilter", "");
        boolean colorErroneousMoves = pars.booleanValue("colorErroneousMoves", false);
        boolean debug = pars.booleanValue("debug", false);

        var results = lsatService.timingAnalysis(projectName, dispatchFilename, settingFilename, goal,
                stochasticSampleLength, exportMotionPlots, motionPlotSampleFrequency, motionPlotFilter,
                colorErroneousMoves, debug);

        response.setStatus(HttpServletResponse.SC_OK);
        response.setContentType(APPLICATION_JSON);
        response.getWriter().write(results);
    }

    private static class Pararameters {
        private final HttpServletRequest request;

        public Pararameters(HttpServletRequest request) {
            this.request = request;
        }

        private String stringValue(String name, String defValue) {
            var result = request.getParameter(name);
            if (result != null) {
                return result;
            }
            return defValue;
        }

        private boolean booleanValue(String name, boolean defValue) {
            var result = request.getParameter(name);
            if (result != null) {
                return Boolean.valueOf(result);
            }
            return defValue;
        }

        private int intValue(String name, int defValue) {
            var result = request.getParameter(name);
            if (result != null) {
                return Integer.parseInt(result);
            }
            return defValue;
        }

        private String stringValue(String name) {
            var result = request.getParameter(name);
            if (result != null) {
                return result;
            }
            throw new IllegalArgumentException("Mandatory argument '" + name + "' not specified");
        }

        private boolean booleanValue(String name) {
            return Boolean.valueOf(stringValue(name));
        }

        private int intValue(String name) {
            return Integer.parseInt(stringValue(name));
        }
    }

    private static void error(HttpServletResponse response, Exception e) throws IOException {
        response.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
        response.setContentType(TEXT_PLAIN);
        try (PrintWriter writer = response.getWriter()) {
            writer.write("Error: " + e.getMessage() + "\n");
            e.printStackTrace(writer);
        }
    }
}
