//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.datasynth.options;

import java.util.Arrays;
import java.util.EnumSet;
import java.util.Set;
import java.util.stream.Collectors;

import org.eclipse.escet.cif.datasynth.settings.CifDataSynthesisSettingsDefaults;
import org.eclipse.escet.cif.datasynth.settings.SynthesisStatistics;
import org.eclipse.escet.common.app.framework.options.EnumSetOption;
import org.eclipse.escet.common.app.framework.options.Options;
import org.eclipse.escet.common.app.framework.output.OutputProvider;

/** Synthesis statistics option. */
public class SynthesisStatisticsOption extends EnumSetOption<SynthesisStatistics> {
    /** Constructor for the {@link SynthesisStatisticsOption} class. */
    public SynthesisStatisticsOption() {
        super(
                // name
                "Statistics",

                // description
                "The kinds of statistics to print. Specify comma separated names of the statistics. " +

                        "Specify \"bdd-gc-collect\" for BDD garbage collection statistics, "
                        + "\"bdd-gc-resize\" for BDD node table resize statistics, "
                        + "\"bdd-perf-node-op\" for BDD node and operation statistics, "
                        + "\"bdd-perf-cont\" for continuous BDD performance statistics, "
                        + "\"bdd-perf-max-nodes\" for maximum used BDD nodes statistics, "
                        + "\"ctrl-sys-states\" for controlled system states statistics, "
                        + "\"timing\" for timing statistics, and/or "
                        + "\"max-memory\" for maximum used memory statistics. "

                        + "By default, no statistics are printed. "
                        + "Prefix a name with \"+\" to add it on top of the defaults, "
                        + "or with \"-\" to remove it from the defaults.",

                // cmdShort
                null,

                // cmdLong
                "stats",

                // cmdValue
                "STATS",

                // defaultValue
                CifDataSynthesisSettingsDefaults.SYNTHESIS_STATISTICS_DEFAULT.clone(),

                // showInDialog
                true,

                // optDialogDescr
                "The kinds of statistics to print.",

                // enumClass
                SynthesisStatistics.class);
    }

    @Override
    public EnumSet<SynthesisStatistics> parseValue(String optName, String valuesTxt) {
        // Detect old syntax and map it to the new syntax, warning users if they use the old syntax.
        String[] parts = valuesTxt.split(",");
        for (int partIdx = 0; partIdx < parts.length; partIdx++) {
            if (parts[partIdx].strip().equals("bdd-perf-cache")) {
                OutputProvider.warn("Statistic \"bdd-perf-cache\" is deprecated and will be removed in a future "
                        + "release. Use its new name \"bdd-perf-node-op\" instead.");
                parts[partIdx] = "bdd-perf-node-op";
                valuesTxt = Arrays.stream(parts).collect(Collectors.joining(","));
            }
        }

        // Parse using the new syntax.
        return super.parseValue(optName, valuesTxt);
    }

    @Override
    protected String getDialogText(SynthesisStatistics stat) {
        switch (stat) {
            case BDD_GC_COLLECT:
                return "BDD garbage collection";
            case BDD_GC_RESIZE:
                return "BDD node table resize";
            case BDD_PERF_NODE_OP:
                return "BDD node and operation";
            case BDD_PERF_CONT:
                return "Continuous BDD performance";
            case BDD_PERF_MAX_NODES:
                return "Maximum used BDD nodes";
            case CTRL_SYS_STATES:
                return "Controlled system states";
            case TIMING:
                return "Timing";
            case MAX_MEMORY:
                return "Maximum used memory";
        }
        throw new RuntimeException("Unknown statistic: " + stat);
    }

    /**
     * Returns the statistics to print.
     *
     * @return The statistics to print.
     */
    public static Set<SynthesisStatistics> getStatistics() {
        return Options.get(SynthesisStatisticsOption.class);
    }
}
